function simulate_swing_arm_and_uniformity()
    % Simulates a pivot swing arm and calculates Christiansen's Uniformity Coefficient (CU).

    %% Parameters
    fieldSize = 1000;          % Field size (meters)
    pivotCenter = [500, 500];  % Pivot center position (meters)
    pivotRadius = 500;         % Pivot radius (meters)
    armLength = 200;           % Swing arm length (meters)
    dt = 0.1;                  % Time step (seconds)
    numSteps = 3600;           % Total number of simulation steps
    theta_deg = linspace(0, 360, numSteps); % Rotation angles in degrees

    %% Preallocate arrays
    xTower = zeros(1, numSteps);
    yTower = zeros(1, numSteps);
    xTip = zeros(1, numSteps);
    yTip = zeros(1, numSteps);
    applied_water = rand(1, numSteps) * 50 + 50; % Simulated water data for CU

    %% Calculate positions
    for k = 1:numSteps
        theta = theta_deg(k);
        xTower(k) = pivotCenter(1) + pivotRadius * cosd(theta);
        yTower(k) = pivotCenter(2) + pivotRadius * sind(theta);
        xTip(k) = xTower(k) + armLength * cosd(theta);
        yTip(k) = yTower(k) + armLength * sind(theta);
    end

    %% Calculate CU
    avgWater = mean(applied_water);
    CU = 100 * (1 - sum(abs(applied_water - avgWater)) / (numSteps * avgWater));
    fprintf("Christiansen's Uniformity Coefficient (CU): %.2f%%\n", CU);

    %% Visualization
    visualizeSwingArmAndUniformity(pivotCenter, fieldSize, theta_deg, xTower, yTower, xTip, yTip, applied_water, CU);
end

function visualizeSwingArmAndUniformity(pivotCenter, fieldSize, theta_deg, xTower, yTower, xTip, yTip, applied_water, CU)
    % Visualizes the swing arm movement and water application uniformity.

    %% Setup figure
    figure('Color', 'w', 'Position', [100, 100, 1200, 600]);

    % Subplot 1: Swing arm movement
    subplot(1, 2, 1);
    hold on; axis equal; grid on;
    axis([0 fieldSize 0 fieldSize]);
    title('Swing Arm Movement', 'FontSize', 14);
    xlabel('X (m)', 'FontSize', 12);
    ylabel('Y (m)', 'FontSize', 12);

    % Plot the field boundary
    plot([0, fieldSize, fieldSize, 0, 0], [0, 0, fieldSize, fieldSize, 0], 'k-', 'LineWidth', 1.5);

    % Initialize plots
    towerPlot = plot(xTower(1), yTower(1), 'bo', 'MarkerFaceColor', 'b', 'DisplayName', 'Last Tower');
    tipPlot = plot(xTip(1), yTip(1), 'ro', 'MarkerFaceColor', 'r', 'DisplayName', 'Arm Tip');
    armLine = plot([xTower(1), xTip(1)], [yTower(1), yTip(1)], 'g-', 'LineWidth', 2, 'DisplayName', 'Swing Arm');

    legend('Location', 'best');

    % Animation loop
    for k = 1:length(theta_deg)
        set(towerPlot, 'XData', xTower(k), 'YData', yTower(k));
        set(tipPlot, 'XData', xTip(k), 'YData', yTip(k));
        set(armLine, 'XData', [xTower(k), xTip(k)], 'YData', [yTower(k), yTip(k)]);
        drawnow;
        pause(0.01); % Control animation speed
    end

    % Subplot 2: Water application uniformity
    subplot(1, 2, 2);
    plot(theta_deg, applied_water, 'b-', 'LineWidth', 2);
    hold on;
    yline(mean(applied_water), 'r--', 'LineWidth', 2, 'DisplayName', 'Mean');
    title('Water Application Uniformity', 'FontSize', 14);
    xlabel('Angle (degrees)', 'FontSize', 12);
    ylabel('Applied Water (m³/m²)', 'FontSize', 12);
    legend(['CU = ', num2str(CU, '%.2f'), '%'], 'Location', 'best');
    grid on;
end
